<?php
/*******************************************************************************
 *
 * LEIDEN OPEN VARIATION DATABASE (LOVD)
 *
 * Created     : 2008-09-04
 * Modified    : 2010-02-22
 * For LOVD    : 2.0-25
 *
 * Access      : Public
 * Purpose     : Display some graphical displays and statistics on this gene's
 *               DNA variants.
 *
 * Copyright   : 2004-2010 Leiden University Medical Center; http://www.LUMC.nl/
 * Programmers : Ir. Gerard C.P. Schaafsma <G.C.P.Schaafsma@LUMC.nl>
 *               Ing. Ivo F.A.C. Fokkema <I.F.A.C.Fokkema@LUMC.nl>
 * Last edited : Ing. Ivo F.A.C. Fokkema <I.F.A.C.Fokkema@LUMC.nl>
 *
 *
 * This file is part of LOVD.
 *
 * LOVD is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * LOVD is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with LOVD; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *************/

define('ROOT_PATH', './');
require ROOT_PATH . 'inc-init.php';

function lovd_determineLocation($sDNA) {
    // Function to determine the location of DNA variants (5'ATG, coding, intron, 3'stop)
    if (preg_match("/[0-9]+[\+\-][0-9]+/", $sDNA)) {
        //variant is located in an intron
        return 2;
    } elseif (preg_match("/\-[0-9]+/", $sDNA)) {
        //variant is located before the 5' ATG start codon
        return 0;
    } elseif (preg_match("/\*[0-9]+/", $sDNA)) {
        //variant is located after the 3' stop codon
        return 3;
    } else {
        //variant is located in the coding region
        return 1;
    }
}

$nBarWidth = 600;// Set the width for a 100% bar

// Array with row headers for the final output tables
$aVariants['sub']['header']             = 'substitutions';
$aVariants['del']['header']             = 'deletions';
$aVariants['dup']['header']             = 'duplications';
$aVariants['ins']['header']             = 'insertions';
$aVariants['delins']['header']          = 'insertion/deletions';
$aVariants['inv']['header']             = 'inversions';
$aVariants['twovars']['header']         = '2 variants in 1 allele';
$aVariants['spl']['header']             = 'splice variants';
$aVariants['fs']['header']              = 'frame shifts';
$aVariants['no protein']['header']      = 'no protein variants';
$aVariants['p.X']['header']             = 'nonstop variants';
$aVariants['X']['header']               = 'nonsense';
$aVariants['p.Met']['header']           = 'translation initiation variant';
$aVariants['=']['header']               = 'silent';
$aVariants['complex']['header']         = 'complex';
$aVariants['unknown']['header']         = 'unknown';
$aVariants['r.0']['header']             = 'no RNA produced';
$aVariants['no effect']['header']       = 'no effect';


// create regular expressions
$sFraShift      = "/fs/";                                                       // frame shift
$sPredFraShift  = "/\([a-zA-Z]{1,3}(\d)*[a-zA-Z]*fs/i";                         // frame shift (predicted)

$sNonStop       = "/extX/";                                                     // nonstop
$sPredNonStop   = "/\([a-zA-Z0-9_]*extX/";                                      // nonstop (predicted)

$sNonsense      = "/X/";                                                        // nonsense
$sPredNonsense  = "/\([a-zA-Z]{1,3}(\d)*X/";                                    // nonsense (predicted)

$sNoProtein      = "/p\.0/";                                                     // no translation
$sPredNoProtein  = "/p\.\(0\)[^\?]/";                                                 // no translation (predicted)

$sTransInit     = "/p\.Met/";                                                   // translation initiation
$sPredTransInit = "/p\.\(Met/";                                                 // translation initiation (predicted)

$sSilent        = "/=/";                                                        // silent

$sUnknown       = "/\?|\(|^r\.$|^p\.$/";                                        // unknown

$sSplice        = "/spl/";                                                      // splice variant
$sPredSplice    = "/\(spl/";                                                    // splice variant (predicted)

$sDelIns        = "/del(\w)*ins/";                                              // insertion/deletion
$sPredDelIns    = "/\([a-zA-Z0-9_]*del(\w)*ins/";                               // insertion/deletion (predicted)

$sInv           = "/inv/";                                                      // inversion
$sPredInv       = "/\([a-zA-Z0-9_]*inv/";                                       // inversion

$sIns           = "/ins/";                                                      // insertion
$sPredIns       = "/\([a-zA-Z0-9_]*ins(\d)*/";                                  // insertion

$sDup           = "/dup/";                                                      // duplication
$sPredDup       = "/\([a-zA-Z0-9_]*dup/";                                       // duplication

$sDel           = "/del/";                                                      // deletion
$sPredDel       = "/\([a-zA-Z0-9_]*del(\d)*/";                                  // predicted deletion

$sSub           = "/>/";                                                        // substitution

$sProtSub       = "/(^p\.)?[^\[][a-zA-Z]{1,3}\d+[a-zA-Z]{1,3}/";                // protein substitution
$sPredProtSub   = "/(^p\.)?\([a-zA-Z]{1,3}(\d)+[a-zA-Z]{1,3}\)$/";              // predicted protein substitution

$sComma         = "/,/";                                                        // a comma denotes a complex situation

$sTwoVars       = "/\;/";                                                       // a semicolon denotes 2 mutations in 1 allele
$sPredTwoVars   = "/\([a-zA-Z0-9_]*\;/";                                        // a semicolon denotes 2 mutations in 1 allele

$sNoRNA         = "/r\.0/";                                                     // no RNA produced

$sProtComp      = "/^p\.\(\=\)/";                                               // (=) denotes a complex situation in case of protein (p.)
$sProtUnknown   = "/\?/";                                                       // unknown in case of a protein


if (HAS_AUTH) {
    // If authorized, check for updates.
    require ROOT_PATH . 'inc-upgrade.php';
}

// If no gene selected, present the selection list.
if (!$_SESSION['currdb'] || $_GET['action'] == 'switch_db') {
    lovd_switchDB();
}

// Whether or not a user can see non-public data.
if (lovd_isCurator($_SESSION['currdb'])) {
    define('IS_CURATOR', true);
} else {
    define('IS_CURATOR', false);
}

// Provide gene homepage.
require ROOT_PATH . 'inc-top.php';
lovd_printGeneHeader();
lovd_printHeader('variant_statistics', 'LOVD - Variant statistics summary');

list($nTotal) = mysql_fetch_row(mysql_query('SELECT COUNT(*) FROM ' . TABLE_PAT2VAR . ' AS t1 WHERE t1.symbol = "' . $_SESSION['currdb'] . '" AND ' . (IS_CURATOR? '1=1' : 't1.status >= ' . STATUS_MARKED)));
if (!$nTotal) {
    print('      There are currently no ' . (IS_CURATOR? '' : 'public ') . 'variants in this gene.<BR>' . "\n");
    lovd_printGeneFooter();
    require ROOT_PATH . 'inc-bot.php';
    exit;
}

// To check availability of the Variant/DNA, Variant/RNA, Variant/Protein and Patient/Times_reported columns, we need the CurrDB class.
require ROOT_PATH . 'class/currdb.php';
$_CURRDB = new CurrDB();

// 2009-06-24; 2.0-19; see if you can use exon lengths
$sFileName = ROOT_PATH . 'refseq/' . $_SESSION['currdb'] . '_table.txt';
if (file_exists($sFileName)) {
    // read each line of the file into an array
    $aExonTable = array();
    $i = 0;
    foreach (file($sFileName) as $line) {
        $aLine = explode("\t", $line);
        $aExonTable[$i][0] = $aLine[5];
        if (isset($aLine[6])) {
            $aExonTable[$i][1] = trim($aLine[6]);
        }
        $i++;
    }

    // 2010-02-22; 2.0-25; Remove the headers, to prevent exon numbers as '00' to cause errors.
    if ($aExonTable[0][0] == 'lengthExon' && $aExonTable[0][1] == 'lengthIntron') {
        unset($aExonTable[0]);
    }
}


// 2009-05-05; 2.0-19 added by Gerard: Count the number of variants per exon
if ($_CURRDB->colExists('Variant/Exon')) {
    // Initialize the counter array of DNA variants
    $aCounts = array();

    if ($_CURRDB->colExists('Patient/Times_Reported')) {
        // Use the Times_Reported column to count the number of patients.
        $sQ = 'SELECT v.`Variant/Exon`, SUM(p.`Patient/Times_Reported`) AS sum' . 
              ' FROM ' . TABLE_CURRDB_VARS . ' AS v' .
              ' LEFT JOIN ' . TABLE_PAT2VAR . ' AS p2v USING (variantid)' .
              ' LEFT JOIN ' . TABLE_PATIENTS . ' AS p ON (p2v.patientid = p.patientid)' .
              ' WHERE p2v.symbol = "' . $_SESSION['currdb'] . '"' . (IS_CURATOR? '' : ' AND p2v.status >= ' . STATUS_MARKED) .
              ' GROUP BY v.`Variant/Exon`';
    } else {
        // No Times_Reported column found, consider every patient entry to be one case.
        $sQ = 'SELECT v.`Variant/Exon`, COUNT(p2v.patientid) AS sum' .
              ' FROM ' . TABLE_CURRDB_VARS . ' AS v' .
              ' LEFT JOIN ' . TABLE_PAT2VAR . ' AS p2v USING (variantid)' .
              ' WHERE p2v.symbol = "' . $_SESSION['currdb'] . '"' . (IS_CURATOR? '' : ' AND p2v.status >= ' . STATUS_MARKED) .
              ' GROUP BY v.`Variant/Exon`';
    }
    $qExons = mysql_query($sQ);
    while ($r = mysql_fetch_row($qExons)) {
        $aCounts[$r[0]] = $r[1];
    }

    // 2009-07-14; 2.0-20; store the number of variants per the number of nucleotides in an array
    $bStandExonNum = true;
    if (isset($aExonTable)) {
        foreach ($aCounts as $nExon => $nVariants) {
            // 2009-06-24; 2.0-19; compensate for exon or intron length
            if (preg_match('/^\d+i$/', $nExon) && !empty($aExonTable[intval($nExon)][1]) && is_numeric($aExonTable[intval($nExon)][1])) {
                // alternative intron numbering (e.g. 02i)
                $aFractionVariants[$nExon] = $nVariants/($aExonTable[intval($nExon)][1]);// number of variants per intron length
            // 2010-02-19; 2.0-25; Added intval() to make sure that exon numbers like "05" match also.
            } elseif (preg_match('/^\d+$/', $nExon) && array_key_exists(intval($nExon), $aExonTable)) {
                // proper exon numbering, only digits allowed
                $aFractionVariants[$nExon] = $nVariants/($aExonTable[intval($nExon)][0]);// number of variants per exon length
            } else {
                // non-standard exon numbering
                $aFractionVariants[$nExon] = 0;
                $bStandExonNum = false;
            }
        }
    }
    // After fetching and counting data, print it to the screen.
    // Print percentages in horizontal bars
    print('      <SPAN class="S15"><B>Variants per exon/intron</B></SPAN><BR>' . "\n" .
          '      <TABLE border cellpadding="2">' . "\n" .
          '        <TR>' . "\n" .
          '          <TH>exon</TH>' . "\n" .
          // 2009-07-14; 2.0-20; add column with number of nucleotides when exon lengths available
          (isset($aExonTable) ? '          <TH>variants/length</TH>' . "\n" : '          <TH># variants</TH>' . "\n") .
          '          <TH>' . (isset($aExonTable) ? '' : 'percentage of variants per exon') . '</TH></TR>' . "\n");

    
    // 2009-08-31; 2.0-21; added variable $nBarFraction to simplify the print the red bar statement
    $nBarFraction = '';
    foreach ($aCounts as $nExon => $nVariants) {
        print('        <TR>' . "\n" .
              '          <TD>' . $nExon . '</TD>' . "\n"); //column with exon numbers

        if (isset($aExonTable)) {
            $nBarFraction = $aFractionVariants[$nExon];
            if (preg_match('/^\d+i$/', $nExon) && !empty($aExonTable[intval($nExon)][1]) && is_numeric($aExonTable[intval($nExon)][1])) {
                // column with variants per number of nucleotides in intron
                print('          <TD align="right">' . $nVariants . ' / ' . $aExonTable[intval($nExon)][1] . ' bp</TD>' . "\n");
            // 2010-02-19; 2.0-25; Added intval() to make sure that exon numbers like "05" match also.
            } elseif (preg_match('/^\d+$/', $nExon) && array_key_exists(intval($nExon), $aExonTable)) {
                // column with variants per number of nucleotides in exon
                print('          <TD align="right">' . $nVariants . ' / ' . $aExonTable[intval($nExon)][0] . ' bp</TD>' . "\n");
            } else {
                print('          <TD align="right">' . $nVariants . '<sup>1</sup></TD>' . "\n");
            }
        } else {
            // 2009-02-12; 2.0-16 prevent division by zero
            $nBarFraction = round($nVariants/(array_sum($aCounts)+0.0000001)*100, 2) . '%';
            print('          <TD align="right">' . $nVariants . '</TD>' . "\n");
        }

        print('          <TD><IMG src="' . ROOT_PATH . 'gfx/lovd_summ_blue.png" alt="' . $nBarFraction . '" title="' . $nBarFraction . '" width="' . (isset($aExonTable) ? $nBarFraction*$nBarWidth/max($aFractionVariants) : round($nVariants/(array_sum($aCounts)+0.0000001)*$nBarWidth)) . '" height="15"></TD></TR>' . "\n");
    }
    // Totals row
    if (isset($aExonTable)) {
        print('        </TABLE>' . "\n\n\n\n");
    } else {
        // 2009-06-24; 2.0-19; print only when no exon lengths availabe
        print('        <TR>' . "\n" .
              '          <TD>total</TD>' . "\n" .
              '          <TD align="right">' . array_sum($aCounts) . '</TD>' . "\n" .
              '          <TD><IMG src="' . ROOT_PATH . 'gfx/lovd_summ_blue.png" alt="100%" title="100%" width="' . $nBarWidth . '" height="15"></TD></TR></TABLE>' . "\n");
    }
    if (!$bStandExonNum) {
        print('<sup>1</sup>When exon/intron lengths are not available, only the numbers of variants are given<BR><BR>' . "\n\n\n\n");
    } else {
        print('<BR>' . "\n\n\n\n");
    }
}





// 2009-08-17; 2.0-21  added notification
lovd_showInfoTable('Please note that numbers shown hereafter can deviate from the numbers when you click on a variant link. Reasons for these differences can be that a variant is reported more than once (see # Reported field) or a homozygous variant.', 'warning');

// 2009-09-01; 2.0-21 initialize an array in the session array for storage of variantid's to be used with the variant type links in the tables
$_SESSION['variant_statistics'][$_SESSION['currdb']] = array();

// Counting the DNA variants
if ($_CURRDB->colExists('Variant/DNA')) {
    // Checking the DNA column

    // Initialize arrays for counting locations of DNA variants (5'ATG, coding, intron, 3'stop)
    $aLocationSub       = array(0, 0, 0, 0);    //substitutions
    $aLocationDel       = array(0, 0, 0, 0);    //deletions
    $aLocationDup       = array(0, 0, 0, 0);    //duplications
    $aLocationIns       = array(0, 0, 0, 0);    //insertions
    $aLocationDelIns    = array(0, 0, 0, 0);    //insertions/deletions
    $aLocationInv       = array(0, 0, 0, 0);    //inversions

    // Initialize the counter array of DNA variants
    $aCounts = array();
    $aCounts['sub']     = $aLocationSub;    //substitutions
    $aCounts['del']     = $aLocationDel;    //deletions
    $aCounts['dup']     = $aLocationDup;    //duplications
    $aCounts['ins']     = $aLocationIns;    //insertions
    $aCounts['delins']  = $aLocationDelIns; //insertions/deletions
    $aCounts['inv']     = $aLocationInv;    //inversions
    $aCounts['twovars'] = 0;                //2 variants in 1 allel
    $aCounts['complex'] = 0;                //complex variants
    $aCounts['unknown'] = 0;                //unknown variants

    // Initialize the total sum of the counter array
    $nTotalSum = 0;

    // 2009-12-16; 2.0-24; added v.type
    if ($_CURRDB->colExists('Patient/Times_Reported')) {
        // Use the Times_Reported column to count the number of patients.
        $sQ = 'SELECT v.variantid, v.`Variant/DNA`, v.type, SUM(p.`Patient/Times_Reported`) AS sum' . 
              ' FROM ' . TABLE_CURRDB_VARS . ' AS v' .
              ' LEFT JOIN ' . TABLE_PAT2VAR . ' AS p2v USING (variantid)' .
              ' LEFT JOIN ' . TABLE_PATIENTS . ' AS p ON (p2v.patientid = p.patientid)' .
              ' WHERE p2v.symbol = "' . $_SESSION['currdb'] . '"' . (IS_CURATOR? '' : ' AND p2v.status >= ' . STATUS_MARKED) .
              ' GROUP BY v.variantid';
    } else {
        // No Times_Reported column found, consider every patient entry to be one case.
        $sQ = 'SELECT v.variantid, v.`Variant/DNA`, v.type, COUNT(p2v.patientid) AS sum' .
              ' FROM ' . TABLE_CURRDB_VARS . ' AS v' .
              ' LEFT JOIN ' . TABLE_PAT2VAR . ' AS p2v USING (variantid)' .
              ' WHERE p2v.symbol = "' . $_SESSION['currdb'] . '"' . (IS_CURATOR? '' : ' AND p2v.status >= ' . STATUS_MARKED) .
              ' GROUP BY v.variantid';
    }
    $qDNA = @mysql_query($sQ);
    while (list($nVariantid, $sDNA, $sType, $nCount) = mysql_fetch_row($qDNA)) {
        // 2009-12-16; 2.0-24; added $sType and use that for counting variant types if possible        
        if ($sType) {
            switch ($sType) {
                case 'subst':
                    // variant is a substitution
                    $aCounts['sub'][lovd_determineLocation($sDNA)] += $nCount;
                    $_SESSION['variant_statistics'][$_SESSION['currdb']]['DNA_sub'][] = $nVariantid;
                    break;
                case 'del':
                    // variant is a deletion
                    $aCounts['del'][lovd_determineLocation($sDNA)] += $nCount;
                    $_SESSION['variant_statistics'][$_SESSION['currdb']]['DNA_del'][] = $nVariantid;
                    break;
                case 'ins':
                    // variant is an insertion
                    $aCounts['ins'][lovd_determineLocation($sDNA)] += $nCount;
                    $_SESSION['variant_statistics'][$_SESSION['currdb']]['DNA_ins'][] = $nVariantid;
                    break;
                case 'delins':
                    // variant is an indel
                    $aCounts['delins'][lovd_determineLocation($sDNA)] += $nCount;
                    $_SESSION['variant_statistics'][$_SESSION['currdb']]['DNA_delins'][] = $nVariantid;
                    break;
            }
        } else {
            if (preg_match($sTwoVars, $sDNA)) {
                // two variants in one allele
                $aCounts['twovars'] += $nCount;
                $_SESSION['variant_statistics'][$_SESSION['currdb']]['DNA_twovars'][] = $nVariantid;
            } elseif (preg_match($sDelIns, $sDNA) || (preg_match($sIns, $sDNA) && preg_match($sDel, $sDNA))) {
                // variant is an indel
                $aCounts['delins'][lovd_determineLocation($sDNA)] += $nCount;
                $_SESSION['variant_statistics'][$_SESSION['currdb']]['DNA_delins'][] = $nVariantid;
            } elseif (preg_match($sInv, $sDNA)) {
                // variant is an inversion
                $aCounts['inv'][lovd_determineLocation($sDNA)] += $nCount;
                $_SESSION['variant_statistics'][$_SESSION['currdb']]['DNA_inv'][] = $nVariantid;
            } elseif (preg_match($sIns, $sDNA)) {
                // variant is an insertion
                $aCounts['ins'][lovd_determineLocation($sDNA)] += $nCount;
                $_SESSION['variant_statistics'][$_SESSION['currdb']]['DNA_ins'][] = $nVariantid;
            } elseif (preg_match($sDup, $sDNA) && !preg_match($sDel, $sDNA)) {
                // variant is a duplication (and not a deletion)
                $aCounts['dup'][lovd_determineLocation($sDNA)] += $nCount;
                $_SESSION['variant_statistics'][$_SESSION['currdb']]['DNA_dup'][] = $nVariantid;
            } elseif (preg_match($sDel, $sDNA)) {
                // variant is a deletion
                $aCounts['del'][lovd_determineLocation($sDNA)] += $nCount;
                $_SESSION['variant_statistics'][$_SESSION['currdb']]['DNA_del'][] = $nVariantid;
            } elseif (preg_match($sSub, $sDNA)) {
                // variant is a substitution
                $aCounts['sub'][lovd_determineLocation($sDNA)] += $nCount;
                $_SESSION['variant_statistics'][$_SESSION['currdb']]['DNA_sub'][] = $nVariantid;
            } elseif (preg_match($sUnknown, $sDNA) || !$sDNA) {
                // unknown variant
                $aCounts['unknown'] += $nCount;
                $_SESSION['variant_statistics'][$_SESSION['currdb']]['DNA_unknown'][] = $nVariantid;
            } else {
                // variant is complex
                $aCounts['complex'] += $nCount;
                $_SESSION['variant_statistics'][$_SESSION['currdb']]['DNA_complex'][] = $nVariantid;
            }
        }
    }

    // Calculate the total number of variants (total sum of the counter array)
    foreach ($aCounts as $sVariant => $nVariants) {
        if (is_array($nVariants)) {
                $nTotalSum += array_sum($nVariants);
        } else {
                $nTotalSum += $nVariants;
        }
    }



    // After fetching and counting data, print it to the screen.
    // Table in a fixed order, also print zero values
    // Print percentages in horizontal bars
    print('      <SPAN class="S15"><B>DNA variants</B></SPAN><BR>' . "\n" .
          '      <TABLE border cellpadding="2">' . "\n" .
          '        <TR>' . "\n" .
          '          <TH>variant</TH>' . "\n" .
          '          <TH>number</TH>' . "\n" .
          '          <TH colspan=4>location</TH>' . "\n" .
          '          <TH colspan=4>percentages</TH></TR>' . "\n" .
          '        <TR>' . "\n" .
          '          <TH>&nbsp;</TH>' . "\n" .
          '          <TH>&nbsp;</TH>' . "\n" .
          '          <TH>5\'start</TH>' . "\n" .
          '          <TH>coding</TH>' . "\n" .
          '          <TH>intron</TH>' . "\n" .
          '          <TH>3\'stop</TH><TD></TD></TR>' . "\n");

    $aAbsentVariants = array(); //2009-06-24; 2.0-19; keep track of non-observed variants
    foreach ($aCounts as $sVariant => $nVariants) {
        // Print for each variant type a row. 
        if (is_array($nVariants)) {
            // The substitutions, deletions, duplications, insertions, indels and inversions
            // can be subdivided according to their location (before the 5' start, coding, intron
            // or after the 3' stop
            // First calculate the sum of the number of variants and print the row header and the total number for this variant
            $nVariantsSum = array_sum($nVariants);
            if ($nVariantsSum != 0) {
                //2009-06-24; 2.0-19; print the observed variants only
                print('        <TR>' . "\n" .
                      '          <TD><A href="' . ROOT_PATH . 'variants.php?select_db=' . $sSymbol . '&action=view_all&view=DNA_' . str_replace('/', '', $sVariant) .'">' . $aVariants[$sVariant]['header'] . '</A></TD>' . "\n" .
                      '          <TD align="right">' . $nVariantsSum . '</TD>' . "\n");
                // Now print the numbers for each location
                foreach ($nVariants as $location => $locnumber) {
                    print('          <TD align="right">' . $locnumber . '</TD>' . "\n");
                }
            } else {
                // 2009-06-24; 2.0-19; store non-observed variants
                $aAbsentVariants[] = $aVariants[$sVariant]['header'];
            }
        } else {
            // The other variants
            // First calculate the sum of the number of variants
            $nVariantsSum = $nVariants;
            if ($nVariantsSum != 0) {
                //2009-06-24; 2.0-19; print observed variants
                print('        <TR>' . "\n" .
                      '          <TD><A href="' . ROOT_PATH . 'variants.php?select_db=' . $sSymbol . '&action=view_all&view=DNA_' . str_replace('/', '', $sVariant) .'">' . $aVariants[$sVariant]['header'] . '</A></TD>' . "\n" .
                      '          <TD align="right">' . $nVariantsSum . '</TD>' . "\n");
                // Now print 4 times a '-'
                for ($i = 0; $i < 4; $i++) {
                    print('          <TD align="right">-</TD>' . "\n");
                }
            } else {
                // 2009-06-24; 2.0-19; store non-observed variants
                $aAbsentVariants[] = $aVariants[$sVariant]['header'];
            }
        }
        // Print the bar with a length relative to the total for this variant.
        // 2009-02-12; 2.0-16 prevent division by zero
        $sPercentage = round($nVariantsSum/($nTotalSum+0.0000001)*100, 2) . '%';
        if ($nVariantsSum != 0) {
            print('          <TD><IMG src="' . ROOT_PATH . 'gfx/lovd_summ_blue.png" alt="' . $sPercentage . '" title="' . $sPercentage . '" width="' . round($nVariantsSum/($nTotalSum+0.0000001)*$nBarWidth) . '" height="15"></TD></TR>' . "\n");
        }
    }

    // Totals row
    print('        <TR>' . "\n" .
          '          <TD>totals</TD>' . "\n" .
          '          <TD align="right">' . $nTotalSum . '</TD>' . "\n");
    for ($i = 0; $i < 4; $i++) {
        print('          <TD align="right">' . 
              ($aCounts['sub'][$i] + 
               $aCounts['del'][$i] + 
               $aCounts['dup'][$i] + 
               $aCounts['ins'][$i] + 
               $aCounts['delins'][$i] + 
               $aCounts['inv'][$i]) . 
              '</TD>' . "\n");
    }
    print('          <TD><IMG src="' . ROOT_PATH . 'gfx/lovd_summ_blue.png" alt="100%" title="100%" width="' . $nBarWidth . '" height="15"></TD></TR></TABLE>' . "\n\n\n\n");
    // 2009-06-24; 2.0-19; print non-observed variants
    if (!empty($aAbsentVariants)) {
        print('Variants not observed: ' . implode($aAbsentVariants, ', ') . '<BR><BR>' . "\n\n\n\n");
    } else {
        print('<BR>' . "\n\n\n\n");
    }
}




// Counting the RNA variants
if ($_CURRDB->colExists('Variant/RNA')) {
    $nTotalSum = 0;
    // Checking the RNA column.

    // Initialize the counter array
    $aCounts = array();
    $aCounts['sub']         = 0;    //substitutions
    $aCounts['del']         = 0;    //deletions
    $aCounts['dup']         = 0;    //duplications
    $aCounts['ins']         = 0;    //insertions
    $aCounts['delins']      = 0;    //insertions/deletions
    $aCounts['inv']         = 0;    //inversions
    $aCounts['spl']         = 0;    //splice variants
    $aCounts['twovars']     = 0;    //2 variants in 1 allel
    $aCounts['complex']     = 0;    //complex variants
    $aCounts['unknown']     = 0;    //unknown variants
    $aCounts['no effect']   = 0;    //no effect variants
    $aCounts['r.0']         = 0;    //no RNA produced


    if ($_CURRDB->colExists('Patient/Times_Reported')) {
        // Use the Times_Reported column to count the number of patients.
        $sQ = 'SELECT v.variantid, v.`Variant/RNA`, SUM(p.`Patient/Times_Reported`) AS sum FROM ' . TABLE_CURRDB_VARS . 
              ' AS v LEFT JOIN ' . TABLE_PAT2VAR . ' AS p2v USING (variantid) LEFT JOIN ' . TABLE_PATIENTS . 
              ' AS p ON (p2v.patientid = p.patientid) WHERE p2v.symbol = "' .$_SESSION['currdb'] . '"' . (IS_CURATOR? '' : ' AND p2v.status >= ' . STATUS_MARKED) . 
              ' GROUP BY v.variantid';
    } else {
        // No Times_Reported column found, consider every patient entry to be one case.
        $sQ = 'SELECT v.variantid, v.`Variant/RNA`, COUNT(p2v.patientid) AS sum FROM ' . TABLE_CURRDB_VARS . ' AS v' .
              ' LEFT JOIN ' . TABLE_PAT2VAR . ' AS p2v' .
              ' USING (variantid)' .
              ' WHERE p2v.symbol = "' . $_SESSION['currdb'] . '"' . (IS_CURATOR? '' : ' AND p2v.status >= ' . STATUS_MARKED) .
              ' GROUP BY v.variantid';
    }
    $qRNA = @mysql_query($sQ);


    while (list($nVariantid, $sRNA, $nCount) = mysql_fetch_row($qRNA)) {
        if (preg_match($sUnknown, $sRNA) || !$sRNA) {
            // unknown variant
            $aCounts['unknown'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['RNA_unknown'][] = $nVariantid;
        } elseif (preg_match($sTwoVars, $sRNA)) {
            // two variants in one allele
            $aCounts['twovars'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['RNA_twovars'][] = $nVariantid;
        } elseif (preg_match($sSplice, $sRNA)) {
            // splice variant
            $aCounts['spl'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['RNA_spl'][] = $nVariantid;
        } elseif (preg_match($sComma, $sRNA)) {
            // complex variant
            $aCounts['complex'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['RNA_complex'][] = $nVariantid;
        } elseif (preg_match($sDelIns, $sRNA)) {
            // variant is an indel
            $aCounts['delins'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['RNA_delins'][] = $nVariantid;
        } elseif (preg_match($sInv, $sRNA)) {
            // variant is an inversion
            $aCounts['inv'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['RNA_inv'][] = $nVariantid;
        } elseif (preg_match($sIns, $sRNA)) {
            // variant is an insertion
            $aCounts['ins'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['RNA_ins'][] = $nVariantid;
        } elseif (preg_match($sDel, $sRNA)) {
            // variant is an deletion
            $aCounts['del'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['RNA_del'][] = $nVariantid;
        } elseif (preg_match($sDup, $sRNA)) {
            // variant is an duplication
            $aCounts['dup'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['RNA_dup'][] = $nVariantid;
        } elseif (preg_match($sSub, $sRNA)) {
            // variant is an substitution
            $aCounts['sub'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['RNA_sub'][] = $nVariantid;
        } elseif (preg_match($sSilent, $sRNA)) {
            // variant is an no effect RNA variant
            $aCounts['no effect'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['RNA_no effect'][] = $nVariantid;
        } elseif (preg_match($sNoRNA, $sRNA)) {
            // variant is an no RNA produced variant
            $aCounts['r.0'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['RNA_r.0'][] = $nVariantid;
        } else {
            // complex variant
            $aCounts['complex'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['RNA_complex'][] = $nVariantid;
        }
    }

    // After fetching and counting data, print it to the screen.
    // Print percentages in horizontal bars
    print('      <SPAN class="S15"><B>RNA variants</B></SPAN><BR>' . "\n" .
          '      <TABLE border="1">' . "\n" .
          '        <TR>' . "\n" .
          '          <TH>variant</TH>' . "\n" .
          '          <TH>number</TH>' . "\n" .
          '          <TH>percentages</TH></TR>' . "\n");

    $aAbsentVariants = array(); //2009-06-24; 2.0-19; keep track of non-observed variants
    foreach ($aCounts as $sVariant => $nVariants) {
        // 2009-02-12; 2.0-16 prevent division by zero
        $sPercentage = round($nVariants/(array_sum($aCounts)+0.0000001)*100, 2) . '%';
        if ($nVariants != 0) {
            // 2009-06-24; 2.0-19; print observed variants only
            print('        <TR>' . "\n" .
                '          <TD><A href="' .ROOT_PATH . 'variants.php?select_db=' . $sSymbol . '&action=view_all&view=RNA_' . $sVariant .'">' . $aVariants[$sVariant]['header'] . '</A></TD>' . "\n" .
                  '          <TD align="right">' . $nVariants . '</TD>' . "\n" .
                  '          <TD><IMG src="' . ROOT_PATH . 'gfx/lovd_summ_blue.png" alt="' . $sPercentage . '" title="' . $sPercentage . '" width="' . (round($nVariants/(array_sum($aCounts)+0.0000001)*$nBarWidth)>1?round($nVariants/(array_sum($aCounts)+0.0000001)*$nBarWidth):1) . '" height="15"></TD></TR>' . "\n");
        } else {
            // 2009-06-24; 2.0-19; store non-observed variants
            $aAbsentVariants[] = $aVariants[$sVariant]['header'];
        }

    }

    // Totals row
    print('        <TR>' . "\n" .
          '          <TD>total</TD>' . "\n" .
          '          <TD align="right">' . array_sum($aCounts) . '</TD>' . "\n" .
          '          <TD><IMG src="' . ROOT_PATH . 'gfx/lovd_summ_blue.png" alt="100%" title="100%" width="' . $nBarWidth . '" height="15"></TD></TR></TABLE>' . "\n\n\n\n");
    
    // 2009-06-24; 2.0-19; print non-observed variants
    if (!empty($aAbsentVariants)) {
        print('Variants not observed: ' . implode($aAbsentVariants, ', ') . '<BR><BR>');
    }

}




// Counting the Protein variants
if ($_CURRDB->colExists('Variant/Protein')) {
    $nTotalSum = 0;
    // Checking the Protein column.

    // Initialize the counter arrays, true and predicted
    $aCounts = array();
    $aCountsPred = array();
    $aCounts['sub']             = 0; // substitutions
    $aCountsPred['sub']         = 0; // substitutions (predicted)
    $aCounts['del']             = 0; // deletions
    $aCountsPred['del']         = 0; // deletions (predicted)
    $aCounts['dup']             = 0; // duplications
    $aCountsPred['dup']         = 0; // duplications (predicted)
    $aCounts['ins']             = 0; // insertions
    $aCountsPred['ins']         = 0; // insertions (predicted)
    $aCounts['delins']          = 0; // insertions/deletions
    $aCountsPred['delins']      = 0; // insertions/deletions (predicted)
    $aCounts['twovars']         = 0; // 2 variants in 1 allel
    $aCountsPred['twovars']     = 0; // 2 variants in 1 allel (predicted)
    $aCounts['fs']              = 0; // frame shift
    $aCountsPred['fs']          = 0; // frame shift (predicted)
    $aCounts['no protein']      = 0; // no protein
    $aCountsPred['no protein']  = 0; // predicted no protein (predicted)
    $aCounts['p.X']             = 0; // nonstop
    $aCountsPred['p.X']         = 0; // nonstop (predicted)
    $aCounts['X']               = 0; // nonsense
    $aCountsPred['X']           = 0; // nonsense (predicted)
    $aCounts['p.Met']           = 0; // translation initiation
    $aCountsPred['p.Met']       = 0; // translation initiation (predicted)
    $aCounts['=']               = 0; // silent
    $aCountsPred['=']           = 0; // silent (predicted)
    $aCounts['complex']         = 0; // complex variants
    $aCountsPred['complex']     = 0; // complex variants (predicted)
    $aCounts['unknown']         = 0; // unknown variants
    $aCountsPred['unknown']     = 0; // unknown variants (predicted)
    

    if ($_CURRDB->colExists('Patient/Times_Reported')) {
        // Use the Times_Reported column to count the number of patients.
        $sQ = 'SELECT v.variantid, v.`Variant/Protein`, SUM(p.`Patient/Times_Reported`) AS sum FROM ' . TABLE_CURRDB_VARS . 
              ' AS v LEFT JOIN ' . TABLE_PAT2VAR . ' AS p2v USING (variantid) LEFT JOIN ' . TABLE_PATIENTS . 
              ' AS p ON (p2v.patientid = p.patientid) WHERE p2v.symbol = "' .$_SESSION['currdb'] . 
              '"' . (IS_CURATOR? '' : ' AND p2v.status >= ' . STATUS_MARKED) . 
              ' GROUP BY v.variantid';
    } else {
        // No Times_Reported column found, consider every patient entry to be one case.
        $sQ = 'SELECT v.variantid, v.`Variant/Protein`, COUNT(p2v.patientid) AS sum FROM ' . TABLE_CURRDB_VARS . ' AS v' .
              ' LEFT JOIN ' . TABLE_PAT2VAR . ' AS p2v' .
              ' USING (variantid)' .
              ' WHERE p2v.symbol = "' . $_SESSION['currdb'] . '"' . (IS_CURATOR? '' : ' AND p2v.status >= ' . STATUS_MARKED) .
              ' GROUP BY v.variantid';
    }
    $qProtein = @mysql_query($sQ);
                
    // 2009-08-24; 2.0-21; countings split in true and predicted
    while (list($nVariantid, $sProtein, $nCount) = mysql_fetch_row($qProtein)) {
        if (preg_match($sPredTwoVars, $sProtein)) {
            // two variants in one allele (predicted)
            $aCountsPred['twovars'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_twovars'][] = $nVariantid;
        } elseif (preg_match($sTwoVars, $sProtein)) {
            // two variants in one allele
            $aCounts['twovars'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_twovars'][] = $nVariantid;
        } elseif (preg_match($sComma, $sProtein)) {
            //complex variant
            $aCounts['complex'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_complex'][] = $nVariantid;
        } elseif (preg_match($sProtUnknown, $sProtein) || !$sProtein) {
            // unknown variant
            $aCounts['unknown'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_unknown'][] = $nVariantid;
        } elseif (preg_match($sPredFraShift, $sProtein)) {
            // predicted frame shift variant (predicted)
            $aCountsPred['fs'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_fs'][] = $nVariantid;
        } elseif (preg_match($sFraShift, $sProtein)) {
            // frame shift variant
            $aCounts['fs'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_fs'][] = $nVariantid;
        } elseif (preg_match($sPredNonStop, $sProtein)) {
            // nonstop variant (predicted)
            $aCountsPred['p.X'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_p.X'][] = $nVariantid;
        } elseif (preg_match($sNonStop, $sProtein)) {
            // nonstop variant
            $aCounts['p.X'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_p.X'][] = $nVariantid;
        } elseif (preg_match($sPredNonsense, $sProtein)) {
            // nonsense variant (predicted)
            $aCountsPred['X'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_X'][] = $nVariantid;
        } elseif (preg_match($sNonsense, $sProtein)) {
            // nonsense variant
            $aCounts['X'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_X'][] = $nVariantid;
        } elseif (preg_match($sPredDelIns, $sProtein)) {
            // variant is an indel (predicted)
            $aCountsPred['delins'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_delins'][] = $nVariantid;
        } elseif (preg_match($sDelIns, $sProtein)) {
            // variant is an indel
            $aCounts['delins'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_delins'][] = $nVariantid;
        } elseif (preg_match($sPredDel, $sProtein)) {
            // variant is a deletion (predicted)
            $aCountsPred['del'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_del'][] = $nVariantid;
        } elseif (preg_match($sDel, $sProtein)) {
            // variant is an deletion
            $aCounts['del'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_del'][] = $nVariantid;
        } elseif (preg_match($sPredDup, $sProtein)) {
            // variant is an duplication (predicted)
            $aCountsPred['dup'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_dup'][] = $nVariantid;
        } elseif (preg_match($sDup, $sProtein)) {
            // variant is an duplication
            $aCounts['dup'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_dup'][] = $nVariantid;
        } elseif (preg_match($sPredIns, $sProtein)) {
            // variant is an insertion (predicted)
            $aCountsPred['ins'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_ins'][] = $nVariantid;
        } elseif (preg_match($sIns, $sProtein)) {
            // variant is an insertion
            $aCounts['ins'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_ins'][] = $nVariantid;
        } elseif (preg_match($sPredNoProtein, $sProtein)) {
            // a no protein variant (predicted)
            $aCountsPred['no protein'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_no protein'][] = $nVariantid;
        } elseif (preg_match($sNoProtein, $sProtein)) {
            // a no translation variant
            $aCounts['no protein'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_no protein'][] = $nVariantid;
        } elseif (preg_match($sPredTransInit, $sProtein)) {
            // a predicted translation initiation variant
            $aCountsPred['p.Met'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_p.Met'][] = $nVariantid;
        } elseif (preg_match($sTransInit, $sProtein)) {
            // a translation initiation variant
            $aCounts['p.Met'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_p.Met'][] = $nVariantid;
        } elseif (preg_match($sPredProtSub, $sProtein)) {
            // variant is an predicted substitution
            $aCountsPred['sub'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_sub'][] = $nVariantid;
        } elseif (preg_match($sProtSub, $sProtein, $aMatch)) {
            // variant is an substitution
            $aCounts['sub'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_sub'][] = $nVariantid;
        } elseif (preg_match($sProtComp, $sProtein)) {
            // a complex variant
            $aCounts['complex'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_complex'][] = $nVariantid;
        } elseif (preg_match($sSilent, $sProtein)) {
            // a silent variant
            $aCounts['='] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_='][] = $nVariantid;
        } else {
            //a complex variant
            $aCounts['complex'] += $nCount;
            $_SESSION['variant_statistics'][$_SESSION['currdb']]['Prot_complex'][] = $nVariantid;
        }
    }

    // After fetching and counting data, print it to the screen.
    // Print percentages in horizontal bars
    print('      <SPAN class="S15"><B>Protein variants</B></SPAN><BR>' . "\n" .
          '      <TABLE border="1">' . "\n" .
          '        <TR>' . "\n" .
          '          <TH>variant</TH>' . "\n" .
          '          <TH>number</TH>' . "\n" .
          '          <TH>percentages</TH></TR>' . "\n");

    $aAbsentVariants = array(); //2009-06-24; 2.0-19; keep track of non-observed variants
    $nSum = array_sum($aCounts) + array_sum($aCountsPred);
    foreach ($aCounts as $sVariant => $nVariants) {
        // 2009-02-12; 2.0-16 prevent division by zero
        $nPercentage = round($nVariants/($nSum + 0.0000001) * 100, 2);
        $nPercentagePred = round($aCountsPred[$sVariant]/($nSum + 0.0000001) *100, 2);
        $sPercentageTotal = ($nPercentage + $nPercentagePred) . '%';
        if ($nVariants != 0 || $aCountsPred[$sVariant] != 0) {
            //2009-06-24; 2.0-19; print observed variants only
            //2009-08-24; 2.0-21; print confirmed and predicted variants separately, except for the complex and unknown variants
            print('        <TR>' . "\n" .
                  '          <TD><A href="' . ROOT_PATH . 'variants.php?select_db=' . $sSymbol . '&action=view_all&view=Prot_' . $sVariant .'">' . $aVariants[$sVariant]['header'] . '</A></TD>' . "\n");
            if (!in_array($sVariant, array('unknown', 'complex'))) {
                print('          <TD align="right">' . ($nVariants?'confirmed: ' . $nVariants:'') . ($aCountsPred[$sVariant]?' predicted: ' . $aCountsPred[$sVariant]: '') . '</TD>' . "\n" .
                      '          <TD>' .
                      ($nPercentage ? '<IMG src="' . ROOT_PATH . 'gfx/lovd_summ_blue.png" alt="' . $nPercentage . '%" title="' . $nPercentage . '%" width="' . (round($nVariants/($nSum + 0.0000001) * $nBarWidth, 2)>1?round($nVariants/($nSum + 0.0000001) * $nBarWidth, 2):1) . '" height="15">' : '') .
                      ($nPercentagePred ? '<IMG src="' . ROOT_PATH . 'gfx/lovd_summ_red.png" alt="' . $nPercentagePred . '%" title="' . $nPercentagePred . '%" width="' . (round($aCountsPred[$sVariant]/($nSum + 0.0000001)*$nBarWidth, 2)>1?round($aCountsPred[$sVariant]/($nSum + 0.0000001)*$nBarWidth, 2):1) . '" height="15">' : '') .
                      '          </TD></TR>' . "\n");
            } else {
                print('          <TD align="right">' . $nVariants . '</TD>' . "\n" .
                      '          <TD>' .
                      ($nPercentage ? '<IMG src="' . ROOT_PATH . 'gfx/lovd_summ_blue.png" alt="' . $nPercentage . '%" title="' . $nPercentage . '%" width="' . (round($nVariants/($nSum + 0.0000001) * $nBarWidth, 2)>1?round($nVariants/($nSum + 0.0000001) * $nBarWidth, 2):1) . '" height="15">' : '') .
                      '          </TD></TR>' . "\n");
            }
                
        } else {
            // 2009-06-24; 2.0-19; store non-observed variants
            $aAbsentVariants[] = $aVariants[$sVariant]['header'];
        }
    }
    // Totals row
    print('        <TR>' . "\n" .
          '          <TD>total</TD>' . "\n" .
          '          <TD align="right">' . $nSum . '</TD>' . "\n" .
          '          <TD>' .
          '<IMG src="' . ROOT_PATH . 'gfx/lovd_summ_blue.png" alt="100%" title="100%" width="' . $nBarWidth . '" height="15">' .       '          </TD></TR></TABLE>' . "\n\n\n\n");

    // 2009-06-24; 2.0-19; print non-observed variants
    if (!empty($aAbsentVariants)) {
        print('Variants not observed: ' . implode($aAbsentVariants, ', ') . '<BR><BR>');
    }
    print('<BR>Legend: <IMG src="' . ROOT_PATH . 'gfx/lovd_summ_blue.png" alt="confirmed" title="confirmed" width="45" height="15"> confirmed <IMG src="' . ROOT_PATH . 'gfx/lovd_summ_red.png" alt="predicted" title="predicted" width="45" height="15"> predicted<BR>');
}


lovd_printGeneFooter();
require ROOT_PATH . 'inc-bot.php';
?>
